/*******************************************************************/
/*                                                                 */
/*                      ADOBE CONFIDENTIAL                         */
/*                   _ _ _ _ _ _ _ _ _ _ _ _ _                     */
/*                                                                 */
/* Copyright 2002 Adobe Systems Incorporated                       */
/* All Rights Reserved.                                            */
/*                                                                 */
/* NOTICE:  All information contained herein is, and remains the   */
/* property of Adobe Systems Incorporated and its suppliers, if    */
/* any.  The intellectual and technical concepts contained         */
/* herein are proprietary to Adobe Systems Incorporated and its    */
/* suppliers and may be covered by U.S. and Foreign Patents,       */
/* patents in process, and are protected by trade secret or        */
/* copyright law.  Dissemination of this information or            */
/* reproduction of this material is strictly forbidden unless      */
/* prior written permission is obtained from Adobe Systems         */
/* Incorporated.                                                   */
/*                                                                 */
/*******************************************************************/

/*
**-----------------------------------------------------------------------------
** Effect File Variables
**-----------------------------------------------------------------------------
*/

uniform float4x4	gWorldViewProj : WorldViewProjection; // This matrix will be loaded by the application
uniform float4x4	gWorldViewInverse;
uniform float4x4	gWorld;
uniform float4		gLightVector;
uniform float		gBumpAmount;
uniform float4		gLightColor = float4(0.8,0.8,0.8,1.0);
uniform float		gFleckAmount;
uniform float		gReflectivity;
uniform float		gCurrentParameter;
uniform float		gCurlAngle;
uniform float		gAspectRatio;
uniform float		gAnchorX;
textureCUBE			gEnvTexture;
texture				gFleckTexture;
texture				gVideoTexture;
texture				gBumpTexture; 

/*
**-----------------------------------------
**		Sampler States
**-----------------------------------------
*/
//incoming video texture
sampler Sampler = sampler_state
{
    Texture   = (gVideoTexture);
    MipFilter = LINEAR;
    MinFilter = LINEAR;
    MagFilter = LINEAR;
};


//environment texture loaded to show effect of room
samplerCUBE SamplerEnv = sampler_state
{
    Texture   = (gEnvTexture);
    MipFilter = LINEAR;
    MinFilter = LINEAR;
    MagFilter = LINEAR;
};

//normal texture to show noise/flecks
sampler SamplerFleck = sampler_state
{
    Texture   = (gFleckTexture);
    MipFilter = LINEAR;
    MinFilter = LINEAR;
    MagFilter = LINEAR;
};

sampler SamplerBump = sampler_state
{
    Texture   = (gBumpTexture);
    MipFilter = LINEAR;
    MinFilter = LINEAR;
    MagFilter = LINEAR;
};


/*
**-----------------------------------------------------------------------------
** Vertex Definitions
**-----------------------------------------------------------------------------
** APP_OUTPUT is the structure in which we receive the vertices from the application
*/
struct APP_OUTPUT
{
    float3 mPosition	: POSITION;
	float3 mNormal		: NORMAL;
	float2 mTexCoord	: TEXCOORD0;
	
};

/* 
** Pixel Shader structure declaration
*/

struct VS_OUTPUT 
{
  float4 mHPosition		: POSITION;		// coord position in window
  float2 mTexcoord		: TEXCOORD0;	// wavy or fleck map texture coordinates
  float3 mLightVec		: TEXCOORD1;		// position of light relative to point
  float3 mHalfVec		: TEXCOORD2;		// Blinn halfangle
  float3 mIncidentVec	: TEXCOORD3;	// position of viewer relative to point
  float2 mBumpCoord		: TEXCOORD4;
};

/*
** TEMP_OUT is a temporary structure for the DoCylCurl function
*/
struct TEMP_OUT
{
	float4 mPosition	: POSITION;
	float3 mNormal		: NORMAL0;
	float3 mTangent;
	float3 mBinormal;
};

/*
**------------------------------------------------
**		Cylindrical Curl effect
**------------------------------------------------
*/
TEMP_OUT DoCylCurl( float3 position, uniform float gCurrentParameter )
{
	TEMP_OUT returnVertex;
	float sinVal, cosVal, curlRadius;
	
	float3x3 forwardMat, InvMat;
	
	sincos( gCurlAngle, sinVal, cosVal );
	forwardMat = float3x3( float3(cosVal, sinVal, 0.0), float3(-sinVal, cosVal, 0.0), float3(0,0,1) );
	sincos( -gCurlAngle, sinVal, cosVal );
	InvMat = float3x3( float3(cosVal, sinVal, 0.0), float3(-sinVal, cosVal, 0.0), float3(0,0,1) );
	
	position = mul( forwardMat, position );
	returnVertex.mPosition.xyz = position;
	returnVertex.mPosition.w = 1.0f;
	
	returnVertex.mNormal = float3( 0.0f, 0.0f, 1.0f );
	returnVertex.mBinormal = float3( 0, 1.0, 0 );	
	
	if ( position.x > gAnchorX )
	{
		//radius = 2/pi - factor*(distance to anchor)
		curlRadius = 0.50 - 0.02*( position.x - gAnchorX );
		float angle = ( position.x - gAnchorX )/curlRadius; // dist/Radius of curl
		sincos( angle, sinVal, cosVal );
		returnVertex.mPosition.x = gAnchorX + sinVal*curlRadius; // gAnchorX + R*SinVal
		returnVertex.mNormal = float3( -sinVal, 0, abs(cosVal) );
		returnVertex.mPosition.z = -0.50 + curlRadius*cosVal;
	}
	
	returnVertex.mPosition.xyz = mul( InvMat, returnVertex.mPosition.xyz );
	returnVertex.mBinormal = mul( InvMat, returnVertex.mBinormal );
	returnVertex.mNormal = mul( InvMat, returnVertex.mNormal );
	returnVertex.mTangent = cross( returnVertex.mBinormal, returnVertex.mNormal );
	
	return returnVertex;
}


/*
**-------------------------------------------------------------------------
** PageCurl effect - Vertex Shader
**-------------------------------------------------------------------------
*/
VS_OUTPUT PageCurl_vs(APP_OUTPUT In)
{
    VS_OUTPUT Out;

    // copy texture coordinates
	Out.mTexcoord.xy = In.mTexCoord.xy;
    
    Out.mBumpCoord = Out.mTexcoord;
    
    TEMP_OUT tempVertex = DoCylCurl(float3(In.mPosition.x*gAspectRatio ,In.mPosition.yz) , gCurrentParameter );
	
	// transform vertex position into homogenous clip-space
    Out.mHPosition = mul(gWorldViewProj, tempVertex.mPosition);
	
    // Generate BASIS matrix
	float3x3 rotationMat = { (tempVertex.mTangent), 
							  (tempVertex.mBinormal), 
							  (tempVertex.mNormal) };
	
	// store normalized light vector
    Out.mLightVec = (gLightVector.xyz - mul(gWorld,tempVertex.mPosition.xyz));
    float lightDist = distance( Out.mLightVec, float3(0.0,0.0,0.0) );
    Out.mLightVec /= lightDist;

	//compute the half vector    
    float4 eyePos = float4(0.0, 0.0, 1.5, 0.0);
    Out.mHalfVec = normalize(Out.mLightVec + eyePos);
    
    //compute Incident vector from eye position
    Out.mIncidentVec = normalize( mul(gWorld,tempVertex.mPosition.xyz) - eyePos);

	//bring the computed directions in s-t (texture) space	
	Out.mLightVec = mul( rotationMat, Out.mLightVec );
	Out.mHalfVec = mul(rotationMat, Out.mHalfVec );
	
	return Out;
}

 
/*
**-------------------------------------------------------------------------
** PageCurl effect - pixel Shader 2_0
**-------------------------------------------------------------------------
*/

float4 PageCurl_ps_2_0(VS_OUTPUT In) : COLOR
{   
	float4 outColor, color1, reflectColor;
	float3 normal;
	float diffuse, specular;
	
	color1 = tex2D( Sampler, In.mTexcoord );
	
	//find out the bump normal
	normal = tex2D( SamplerBump, In.mBumpCoord ).xyz;
	normal = 2.0*(normal - 0.5);
	normal = lerp( float3( 0.0,0.0,1.0 ), normal, gBumpAmount );

	diffuse = dot(normal, In.mLightVec );
	
	specular = pow(dot(normal, In.mHalfVec ),32);
	color1.xyz = color1*(diffuse)+ (specular)*color1*gLightColor;

	float3 R = reflect( In.mIncidentVec, normal );
	reflectColor = texCUBE( SamplerEnv, R );
	outColor.xyz = lerp( color1.xyz, reflectColor.xyz, gReflectivity );

	// two sampling on the noise texture to make noise look non-repeating.
	float3 fleckN      = (float3)tex2D(SamplerFleck, In.mTexcoord*19);
	fleckN      = fleckN + (float3)tex2D(SamplerFleck, In.mTexcoord*11) - 1;
	specular = saturate( dot(fleckN, In.mHalfVec ) );
	specular = pow ( specular, 64 );
	
	//put the specular color due to fleck
	outColor.xyz += gFleckAmount*specular*float3(0.6, 0.5, 0.1);
	outColor.a = color1.a;
	
    return outColor;
}
/*
**-------------------------------------------------------------------------
** PageCurl effect - pixel Shader 1_4
**-------------------------------------------------------------------------
*/

float4 PageCurl_ps_1_4(VS_OUTPUT In) : COLOR
{   
	float4 outColor, color1, reflectColor;
	float3 normal;
	float diffuse, specular;
	
	color1 = tex2D( Sampler, In.mTexcoord );
	
	//find out the bump normal
	normal = tex2D( SamplerBump, In.mBumpCoord ).xyz;
	normal = 2.0*(normal - 0.5);
	normal = lerp( float3( 0.0,0.0,1.0 ), normal, gBumpAmount );

	diffuse = dot(normal, In.mLightVec);
	specular = In.mHalfVec.z;
	color1.xyz = color1*(diffuse + specular*specular*0.25);

	float3 R = reflect( In.mIncidentVec, normal );
	reflectColor = texCUBE( SamplerEnv, R );
	outColor.xyz = lerp( color1.xyz, reflectColor.xyz, gReflectivity );
	
	outColor.a = color1.a;
	return outColor;
}
/*
**-------------------------------------------------------------------------
** PageCurl effect - pixel Shader 1_3
**-------------------------------------------------------------------------
*/

float4 PageCurl_ps_1_3(VS_OUTPUT In) : COLOR
{   
	float4 outColor, color1, reflectColor;
	float diffuse, specular;
	
	color1 = tex2D( Sampler, In.mTexcoord );
	
	diffuse = In.mLightVec.z;
	specular = In.mHalfVec.z;
	specular *= specular;
	specular *= specular;
	specular *= specular;
	specular *= specular;
	
	outColor.xyz = color1*(diffuse) + (specular)*color1*float3(0.45,0.45,0.45);
	outColor.a = color1.a;
	
    return outColor;
}

/*-----------------------------------------------------------------------------
**		technique PageCurl_2_0
**-----------------------------------------------------------------------------
*/
technique PageCurl_2_0
{
    pass Pass0
    {
		Sampler[0] = (Sampler); 
		Sampler[1] = (SamplerBump);
		Sampler[2] = (SamplerEnv);
		Sampler[3] = (SamplerFleck);
		
		VertexShader = compile vs_2_0 PageCurl_vs();
		PixelShader  = compile ps_2_0 PageCurl_ps_2_0();
    }
}

technique PageCurl_1_4
{
    pass Pass0
    {
		Sampler[0] = (Sampler); 
		Sampler[1] = (SamplerBump);
		Sampler[2] = (SamplerEnv);
				
		VertexShader = compile vs_1_1 PageCurl_vs();
		PixelShader  = compile ps_1_4 PageCurl_ps_1_4();
    }
}

technique PageCurl_1_3
{
    pass Pass0
    {
		Sampler[0] = (Sampler); 
		
		VertexShader = compile vs_1_1 PageCurl_vs();
		PixelShader  = compile ps_1_3 PageCurl_ps_1_3();
    }
}
